<?php
use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Carbon\Carbon;

class LaratrustSetupTables extends Migration
{
    /**
     * Run the migrations.
     *
     * @return  void
     */
    public function up()
    {
        // Create table for storing roles
        Schema::create('roles', function (Blueprint $table) {
            $table->increments('id');
            $table->string('name')->unique();
            $table->string('display_name')->nullable();
            $table->string('description')->nullable();
            $table->timestamps();
        });

        // Create table for storing permissions
        Schema::create('permissions', function (Blueprint $table) {
            $table->increments('id');
            $table->string('name')->unique();
            $table->string('display_name')->nullable();
            $table->string('description')->nullable();
            $table->timestamps();
            $table->integer('groups')->nullable()->default(0);
            $table->integer('order_by')->nullable()->default(0);
        });

        // Create table for associating roles to users and teams (Many To Many Polymorphic)
        Schema::create('role_user', function (Blueprint $table) {
            $table->unsignedInteger('role_id');
            $table->unsignedInteger('user_id');
            $table->string('user_type');

            $table->foreign('role_id')->references('id')->on('roles')
                ->onUpdate('cascade')->onDelete('cascade');

            $table->primary(['user_id', 'role_id', 'user_type']);
        });

        // Create table for associating permissions to users (Many To Many Polymorphic)
        Schema::create('permission_user', function (Blueprint $table) {
            $table->unsignedInteger('permission_id');
            $table->unsignedInteger('user_id');
            $table->string('user_type');

            $table->foreign('permission_id')->references('id')->on('permissions')
                ->onUpdate('cascade')->onDelete('cascade');

            $table->primary(['user_id', 'permission_id', 'user_type']);
        });

        // Create table for associating permissions to roles (Many-to-Many)
        Schema::create('permission_role', function (Blueprint $table) {
            $table->unsignedInteger('permission_id');
            $table->unsignedInteger('role_id');

            $table->foreign('permission_id')->references('id')->on('permissions')
                ->onUpdate('cascade')->onDelete('cascade');
            $table->foreign('role_id')->references('id')->on('roles')
                ->onUpdate('cascade')->onDelete('cascade');

            $table->primary(['permission_id', 'role_id']);
        });


        $current_time = \Carbon\Carbon::now()->toDateTimeString();
        DB::table('roles')->insert([
            [
                'name' => "super-admin",
                'display_name' => "Super Admin",
                'created_at' => $current_time,
                'updated_at' => $current_time,
            ],
            [
                'name' => "admin",
                'display_name' => "Admin",
                'created_at' => $current_time,
                'updated_at' => $current_time,
            ]
        ]);

        DB::table('role_user')->insert([
            [
                'role_id' => 1,
                'user_id' => 1,
                'user_type' => "App\User"
            ],
            [
                'role_id' => 1,
                'user_id' => 2,
                'user_type' => "App\User"
            ],
            [
                'role_id' => 2,
                'user_id' => 3,
                'user_type' => "App\User"
            ]
        ]);


        DB::table('permissions')->insert([
            [
                'name' => "view-user-list",
                'display_name' => "View user list",
                'created_at' => $current_time,
                'updated_at' => $current_time,
                'groups' => 1,
                'order_by' => 1,
            ],
            [
                'name' => "create-user",
                'display_name' => "Create user",
                'created_at' => $current_time,
                'updated_at' => $current_time,
                'groups' => 1,
                'order_by' => 2,
            ],
            [
                'name' => "edit-user",
                'display_name' => "Edit user",
                'created_at' => $current_time,
                'updated_at' => $current_time,
                'groups' => 1,
                'order_by' => 3,
            ],
            [
                'name' => "view-role-list",
                'display_name' => "View role list",
                'created_at' => $current_time,
                'updated_at' => $current_time,
                'groups' => 2,
                'order_by' => 4,
            ],
            [
                'name' => "create-role",
                'display_name' => "Create role",
                'created_at' => $current_time,
                'updated_at' => $current_time,
                'groups' => 2,
                'order_by' => 5,
            ],
            [
                'name' => "edit-role",
                'display_name' => "Edit role",
                'created_at' => $current_time,
                'updated_at' => $current_time,
                'groups' => 2,
                'order_by' => 6,
            ]
        ]);



        DB::table('permission_role')->insert([
            [
                'permission_id' => "1",
                'role_id' => "1",
            ],
            [
                'permission_id' => "2",
                'role_id' => "1",
            ],
            [
                'permission_id' => "3",
                'role_id' => "1",
            ],
            [
                'permission_id' => "4",
                'role_id' => "1",
            ],
            [
                'permission_id' => "5",
                'role_id' => "1",
            ],
            [
                'permission_id' => "6",
                'role_id' => "1",
            ],
            [
                'permission_id' => "1",
                'role_id' => "2",
            ],
            [
                'permission_id' => "2",
                'role_id' => "2",
            ],
            [
                'permission_id' => "3",
                'role_id' => "2",
            ],
            [
                'permission_id' => "4",
                'role_id' => "2",
            ]
        ]);
    }

    /**
     * Reverse the migrations.
     *
     * @return  void
     */
    public function down()
    {
        Schema::dropIfExists('permission_role');
        Schema::dropIfExists('permissions');
        Schema::dropIfExists('roles');
    }
}
